import { EnhancedEventEmitter } from './EnhancedEventEmitter';
import { Router, RouterOptions } from './Router';
import { WebRtcServer, WebRtcServerOptions } from './WebRtcServer';
import { AppData } from './types';
export type WorkerLogLevel = 'debug' | 'warn' | 'error' | 'none';
export type WorkerLogTag = 'info' | 'ice' | 'dtls' | 'rtp' | 'srtp' | 'rtcp' | 'rtx' | 'bwe' | 'score' | 'simulcast' | 'svc' | 'sctp' | 'message';
export type WorkerSettings<WorkerAppData extends AppData = AppData> = {
    /**
     * Logging level for logs generated by the media worker subprocesses (check
     * the Debugging documentation). Valid values are 'debug', 'warn', 'error' and
     * 'none'. Default 'error'.
     */
    logLevel?: WorkerLogLevel;
    /**
     * Log tags for debugging. Check the meaning of each available tag in the
     * Debugging documentation.
     */
    logTags?: WorkerLogTag[];
    /**
     * Minimun RTC port for ICE, DTLS, RTP, etc. Default 10000.
     */
    rtcMinPort?: number;
    /**
     * Maximum RTC port for ICE, DTLS, RTP, etc. Default 59999.
     */
    rtcMaxPort?: number;
    /**
     * Path to the DTLS public certificate file in PEM format. If unset, a
     * certificate is dynamically created.
     */
    dtlsCertificateFile?: string;
    /**
     * Path to the DTLS certificate private key file in PEM format. If unset, a
     * certificate is dynamically created.
     */
    dtlsPrivateKeyFile?: string;
    /**
     * Field trials for libwebrtc.
     * @private
     *
     * NOTE: For advanced users only. An invalid value will make the worker crash.
     * Default value is
     * "WebRTC-Bwe-AlrLimitedBackoff/Enabled/".
     */
    libwebrtcFieldTrials?: string;
    /**
     * Custom application data.
     */
    appData?: WorkerAppData;
};
export type WorkerUpdateableSettings<T extends AppData = AppData> = Pick<WorkerSettings<T>, 'logLevel' | 'logTags'>;
/**
 * An object with the fields of the uv_rusage_t struct.
 *
 * - http://docs.libuv.org/en/v1.x/misc.html#c.uv_rusage_t
 * - https://linux.die.net/man/2/getrusage
 */
export type WorkerResourceUsage = {
    /**
     * User CPU time used (in ms).
     */
    ru_utime: number;
    /**
     * System CPU time used (in ms).
     */
    ru_stime: number;
    /**
     * Maximum resident set size.
     */
    ru_maxrss: number;
    /**
     * Integral shared memory size.
     */
    ru_ixrss: number;
    /**
     * Integral unshared data size.
     */
    ru_idrss: number;
    /**
     * Integral unshared stack size.
     */
    ru_isrss: number;
    /**
     * Page reclaims (soft page faults).
     */
    ru_minflt: number;
    /**
     * Page faults (hard page faults).
     */
    ru_majflt: number;
    /**
     * Swaps.
     */
    ru_nswap: number;
    /**
     * Block input operations.
     */
    ru_inblock: number;
    /**
     * Block output operations.
     */
    ru_oublock: number;
    /**
     * IPC messages sent.
     */
    ru_msgsnd: number;
    /**
     * IPC messages received.
     */
    ru_msgrcv: number;
    /**
     * Signals received.
     */
    ru_nsignals: number;
    /**
     * Voluntary context switches.
     */
    ru_nvcsw: number;
    /**
     * Involuntary context switches.
     */
    ru_nivcsw: number;
};
export type WorkerEvents = {
    died: [Error];
    '@success': [];
    '@failure': [Error];
};
export type WorkerObserverEvents = {
    close: [];
    newwebrtcserver: [WebRtcServer];
    newrouter: [Router];
};
export declare const workerBin: string;
export declare class Worker<WorkerAppData extends AppData = AppData> extends EnhancedEventEmitter<WorkerEvents> {
    #private;
    /**
     * @private
     */
    constructor({ logLevel, logTags, rtcMinPort, rtcMaxPort, dtlsCertificateFile, dtlsPrivateKeyFile, libwebrtcFieldTrials, appData }: WorkerSettings<WorkerAppData>);
    /**
     * Worker process identifier (PID).
     */
    get pid(): number;
    /**
     * Whether the Worker is closed.
     */
    get closed(): boolean;
    /**
     * Whether the Worker died.
     */
    get died(): boolean;
    /**
     * App custom data.
     */
    get appData(): WorkerAppData;
    /**
     * App custom data setter.
     */
    set appData(appData: WorkerAppData);
    /**
     * Observer.
     */
    get observer(): EnhancedEventEmitter<WorkerObserverEvents>;
    /**
     * @private
     * Just for testing purposes.
     */
    get webRtcServersForTesting(): Set<WebRtcServer>;
    /**
     * @private
     * Just for testing purposes.
     */
    get routersForTesting(): Set<Router>;
    /**
     * Close the Worker.
     */
    close(): void;
    /**
     * Dump Worker.
     */
    dump(): Promise<any>;
    /**
     * Get mediasoup-worker process resource usage.
     */
    getResourceUsage(): Promise<WorkerResourceUsage>;
    /**
     * Update settings.
     */
    updateSettings({ logLevel, logTags }?: WorkerUpdateableSettings<WorkerAppData>): Promise<void>;
    /**
     * Create a WebRtcServer.
     */
    createWebRtcServer<WebRtcServerAppData extends AppData = AppData>({ listenInfos, appData }: WebRtcServerOptions<WebRtcServerAppData>): Promise<WebRtcServer<WebRtcServerAppData>>;
    /**
     * Create a Router.
     */
    createRouter<RouterAppData extends AppData = AppData>({ mediaCodecs, appData }?: RouterOptions<RouterAppData>): Promise<Router<RouterAppData>>;
    private workerDied;
}
//# sourceMappingURL=Worker.d.ts.map