// segmentationWorker.js
self.importScripts('https://cdn.jsdelivr.net/npm/@mediapipe/selfie_segmentation/selfie_segmentation.js');

const selfieSegmentation = new SelfieSegmentation({
    locateFile: (file) => {
        // Map specific files to their CDN paths
        if (file === 'selfie_segmentation.tflite' || file === 'selfie_segmentation_landscape.tflite') {
            return `https://cdn.jsdelivr.net/npm/@mediapipe/selfie_segmentation/${file}`;
        }
        return `https://cdn.jsdelivr.net/npm/@mediapipe/selfie_segmentation/${file}`;
    }
});

selfieSegmentation.setOptions({ modelSelection: 1, selfieMode: true });

let segmentationCanvas, segmentationCtx;

self.onmessage = async (event) => {
    const { type, imageData, width, height, vbType, bgImageData } = event.data;

    if (type === 'init') {
        segmentationCanvas = new OffscreenCanvas(width, height);
        segmentationCtx = segmentationCanvas.getContext('2d');
        segmentationCtx.imageSmoothingEnabled = true;
        segmentationCtx.imageSmoothingQuality = 'high';
        try {
            await selfieSegmentation.initialize();
            self.postMessage({ type: 'ready' });
        } catch (err) {
            self.postMessage({ type: 'error', message: 'Failed to initialize SelfieSegmentation: ' + err.message });
        }
        return;
    }

    if (!segmentationCanvas || !imageData) return;

    try {
        const bitmap = await createImageBitmap(imageData);
        await selfieSegmentation.send({ image: bitmap });

        selfieSegmentation.onResults(async (results) => {
            if (!results.segmentationMask) {
                self.postMessage({ type: 'error', message: 'No segmentation mask received' });
                return;
            }

            segmentationCtx.clearRect(0, 0, width, height);
            segmentationCtx.save();
            segmentationCtx.scale(-1, 1);
            segmentationCtx.drawImage(bitmap, -width, 0, width, height);
            segmentationCtx.restore();
            segmentationCtx.globalCompositeOperation = 'destination-in';
            segmentationCtx.drawImage(results.segmentationMask, 0, 0, width, height);
            segmentationCtx.globalCompositeOperation = 'source-over';

            const outputCanvas = new OffscreenCanvas(width, height);
            const outputCtx = outputCanvas.getContext('2d');
            outputCtx.imageSmoothingEnabled = true;
            outputCtx.imageSmoothingQuality = 'high';

            outputCtx.clearRect(0, 0, width, height);
            if (vbType === 1) {
                outputCtx.filter = 'blur(1px)';
                outputCtx.drawImage(bitmap, 0, 0, width, height);
                outputCtx.filter = 'none';
            } else if (vbType === 2 && bgImageData) {
                const bgImage = await createImageBitmap(bgImageData);
                outputCtx.drawImage(bgImage, 0, 0, width, height);
            } else {
                outputCtx.fillStyle = 'skyblue';
                outputCtx.fillRect(0, 0, width, height);
            }

            outputCtx.filter = 'contrast(1.1) brightness(1.05)';
            outputCtx.drawImage(segmentationCanvas, 0, 0, width, height);
            outputCtx.filter = 'none';

            const processedData = outputCtx.getImageData(0, 0, width, height);
            self.postMessage({ type: 'frame', processedData, width, height }, [processedData.data.buffer]);
        });
    } catch (err) {
        self.postMessage({ type: 'error', message: 'Processing failed: ' + err.message });
    }
};